<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2024 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\models;

use app\helpers\ArrayHelper;
use app\helpers\SubmissionHelper;
use Yii;
use yii\behaviors\TimestampBehavior;
use yii\db\ActiveRecord;
use yii\helpers\Json;

/**
 * This is the model class for table "form_data".
 *
 * @property integer $id
 * @property integer $form_id
 * @property string $builder
 * @property string $fields
 * @property string $html
 * @property integer $height
 * @property string $version
 * @property integer $created_at
 * @property integer $updated_at
 *
 * @property Form $form
 */
class FormData extends ActiveRecord
{
    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return '{{%form_data}}';
    }

    /**
     * @inheritdoc
     */
    public static function primaryKey()
    {
        return ['id'];
    }

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            TimestampBehavior::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['form_id', 'height'], 'required'],
            [['form_id', 'height', 'created_at', 'updated_at'], 'integer'],
            [['builder','fields','html', 'version'], 'string']
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => Yii::t('app', 'ID'),
            'form_id' => Yii::t('app', 'Form ID'),
            'builder' => Yii::t('app', 'Form Builder'),
            'fields' => Yii::t('app', 'Form Fields'),
            'html' => Yii::t('app', 'Form Html'),
            'height' => Yii::t('app', 'Height'),
            'version' => Yii::t('app', 'Version'),
            'created_at' => Yii::t('app', 'Created At'),
            'updated_at' => Yii::t('app', 'Updated At'),
        ];
    }

    /**
     * @inheritdoc
     */
    public function beforeSave($insert)
    {

        if (!parent::beforeSave($insert)) {
            return false;
        }

        if ($insert) {
            $this->version = Yii::$app->version;
        }

        return true;
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getForm()
    {
        return $this->hasOne(Form::class, ['id' => 'form_id']);
    }

    /**
     * Get Form Name
     *
     * @return mixed|string|null
     */
    public function getFormName()
    {
        $builder = Json::decode($this->builder, true);

        return $builder['settings']['name'] ?? null;
    }

    /**
     * Set Form Name
     *
     * @param $name
     */
    public function setFormName($name)
    {
        $builder = Json::decode($this->builder, true);
        if (isset($builder['settings']['name'])) {
            $builder['settings']['name'] = $name;
            $this->builder = Json::htmlEncode($builder);
        }
    }

    /**
     * Return Form Styles
     *
     * @return array
     * @throws \Exception
     */
    public function getStyles(): array
    {
        $builder = Json::decode($this->builder, true);
        return ArrayHelper::getValue($builder, 'styles') ?? [];
    }

    /**
     * Return the label of a field
     *
     * Used only inside this model
     *
     * @param array $field Item saved in fields attribute
     * @param bool|bool $withAlias Used to append an alias to identify the field
     * @param bool|bool $isGroup Used by radio and checkbox components
     * @param bool|bool $isMatrix Used by radio components
     * @return string
     */
    private function getFieldLabel(array $field, bool $withAlias = false, bool $isGroup = false, bool $isMatrix = false): string
    {
        if ($isMatrix) {
            if (isset($field['data-matrix-label'], $field['data-matrix-question'], $field['data-matrix-answer'])) {
                if (isset($field['groupLabel'])) {
                    $label = $field['data-matrix-label'] . " > " . $field['data-matrix-question'];
                } else {
                    $label = $field['data-matrix-label'] . " > " . $field['data-matrix-question'] . " > " . $field['data-matrix-answer'];
                }
            } else {
                $label = Yii::t('app', 'Matrix of') . " " . $field['data-matrix-id'];
            }
        } elseif ($isGroup) {
            if (isset($field['data-matrix-label'], $field['data-matrix-question'])) {
                // For matrix fields, replace label by answer
                $label = $field['data-matrix-label'] . " > " . $field['data-matrix-question'];
            } else if (isset($field['groupLabel'])) {
                $label = $field['groupLabel'];
            } else {
                $label = Yii::t('app', 'Group of') . " " . $field['name'];
            }
        } else {
            if (!empty($field['data-repeater-label']) && !empty($field['label'])) {
                $label = $field['label'];
            } elseif (!empty($field['label'])) {
                $label = $field['label'];
            } elseif (!empty($field['data-label'])) {
                $label = $field['data-label'];
            } elseif (isset($field['type'], $field['value']) && $field['type'] == 'button') {
                // For buttons, replace label by value
                $label = $field['value'];
            } elseif (isset($field['data-matrix-label'], $field['data-matrix-question'], $field['data-matrix-answer'])) {
                // For matrix fields, replace label by answer
                $label = $field['data-matrix-label'] . " > " . $field['data-matrix-question'] . " > " . $field['data-matrix-answer'];
            } else {
                $label = Yii::t('app', 'Field with ID') . ": " . $field['id'];
                $withAlias = true;
            }
        }

        // Append 'alias'
        if ($withAlias && !empty($field['alias'])) {
            $label .= ' (' . $field['alias'] . ')';
        }

        return $label;
    }

    /**
     * Return the type of field
     *
     * Used only inside this model
     *
     * @param $field
     * @return string|null
     */
    public function getFieldType($field): ?string
    {
        $type = null;

        if (substr($field['name'], 0, 16) === 'hidden_signature') {
            $type = 'signature';
        } else if (!empty($field['name'])) {
            $parts = explode('_', $field['name']);
            $type = !empty($parts[0]) ? $parts[0] : null;
        }

        return $type;
    }

    /**
     * Get Name and Label of Provided Fields
     * Format: [name => label]
     *
     * @param array $fields
     * @param bool $withAlias
     * @return array
     */
    public function getNameAndLabelOfProvidedFields(array $fields = [], bool $withAlias = false): array
    {
        if (empty($fields)) {
            return $fields;
        }

        $options = array();

        foreach ($fields as $field) {

            // For Checkboxes and Radio Buttons,
            // We take the name and label of the group
            if (isset($field['type']) && ( $field['type'] === "checkbox" || $field['type'] === "radio")) {

                // Check if the field was saved in options before
                if (isset($field['name']) && isset($options[$field['name']])) {
                    continue;
                }

                // Set option attributes
                $option = [
                    'name' => $field['name'],
                    'label' => $this->getFieldLabel($field, $withAlias, true), // Get groupLabel,
                ];

                // Save the option, by the name of the field
                $options[$option['name']] = $option;

                continue; // Skip the rest of the current loop iteration

            }

            // Default
            $option = [
                'name' => $field['id'] ?? '',
                'label' => $this->getFieldLabel($field, $withAlias, false),
            ];
            $options[$option['name']] = $option;
        }

        return array_values($options); // Remove keys
    }

    /**
     * Return Fields in rule variables format.
     * Required for rule builder
     *
     * Note: Radios are grouped by its name and buttons are excluded
     *
     * @param bool $withForm
     * @param bool $withAlias
     * @return array
     */
    public function getRuleVariables(bool $withForm = true, bool $withAlias = true): array
    {
        $allFields = Json::decode($this->fields, true);

        // Exclude submit/image/reset buttons.
        $fields = ArrayHelper::exclude($allFields, ['submit', 'reset', 'image'], 'type');
        // Radios
        $radioFields = ArrayHelper::filter($fields, 'radio', 'type');
        // Group Radios by name
        $radioGroups = ArrayHelper::group($radioFields, 'name');

        $variables = array();

        foreach ($fields as $field) {

            // For Radio Buttons,
            // We take the name and label of the group
            // and options item save the value of each radio
            if (isset($field['type']) && $field['type'] === "radio") {

                // Check if the radio group was saved in variables before
                if (isset($field['name']) && isset($variables[$field['name']])) {
                    continue;
                }

                // Get all radios with the same name
                $radios = $radioGroups[$field['name']] ?? null;

                // Get first radio
                $firstRadio = ArrayHelper::first($radios);

                // Set variable attributes
                $variable = [
                    'name' => $firstRadio['name'],
                    'label' => $this->getFieldLabel($firstRadio, $withAlias, true), // Get groupLabel
                    'fieldType' => 'radio',
                ];

                // Get each radio value and add to options
                $options = [];
                foreach ($radios as $radio) {
                    $option = [
                        "value" => $radio['value'],
                        "label" => $this->getFieldLabel($radio, $withAlias, false),
                    ];
                    $options[] = $option;
                }

                if (count($options) > 0) {
                    $variable['options'] = $options;
                }

                // Save the variable, by the name of the radio group
                $variables[$variable['name']] = $variable;

                continue; // Skip the rest of the current loop iteration
            } elseif (isset($field['type']) && $field['type'] === "hidden") {
                // Check if the hidden field stores a signature
                if (!empty($field['name']) && substr($field['name'], 0, 16) === "hidden_signature") {
                    $field['type'] = "signature";
                }
            }

            $variable = [
                'name' => $field['id'] ?? '', // Because multiple checkboxes may have the same name
                'label' => $this->getFieldLabel($field, $withAlias, false),
                'fieldType' => $field['type'] ?? $field['tagName'],
            ];

            $options = [];

            if (isset($field['options'])) { // Select List has options
                foreach ($field['options'] as $opt) {
                    $label = $opt['label'] ?? '';
                    $option = [
                        "value" => $opt['value'] ?? "",
                        "label" => isset($opt['data-group-label']) ? $opt['data-group-label'] . ' > ' . $label : $label,
                    ];
                    $options[] = $option;
                }
            }

            if (count($options) > 0) {
                $variable['options'] = $options;
            }

            $variables[$variable['name']] = $variable;
        }

        if ($withForm) {
            // Add Form to variables
            $form = [
                'name' => "form",
                'label' => Yii::t('app', "This form"),
                'fieldType' => "form",
            ];

            $variables[$form['name']] = $form;
        }

        return array_values($variables); // Remove keys
    }

    /**
     * Return Fields in rule actions format.
     * Required for rule builder
     *
     * Note: Radios and Checkboxes are grouped by groupLabel and name
     *
     * @param bool $withAlias
     * @return array
     */
    public function getRuleFields(bool $withAlias = true): array
    {
        $fields = Json::decode($this->fields, true);

        $options = array();

        foreach ($fields as $field) {

            // For Repeater Fields
            if (!empty($field['data-repeater-id'])) {

                // Check if the field was saved in options before
                if (isset($field['name']) && isset($options[$field['name']])) {
                    continue;
                }

                // Set option attributes
                $label =  $field['label'] ?? '';
                $option = [
                    'name' => $field['data-repeater-id'],
                    'label' => $field['data-repeater-label'] ?? $label,
                ];

                // Save the option, by the name of the field
                $options[$option['name']] = $option;

                continue; // Skip the rest of the current loop iteration

            }

            // For Matrix Fields
            if (!empty($field['data-matrix-id'])) {

                // Check if the field was saved in options before
                if (isset($field['name']) && isset($options[$field['name']])) {
                    continue;
                }

                // Set option attributes
                $label =  $field['label'] ?? '';
                $option = [
                    'name' => $field['data-matrix-id'],
                    'label' => $field['data-matrix-label'] ?? $label,
                ];

                // Save the option, by the name of the field
                $options[$option['name']] = $option;

                continue; // Skip the rest of the current loop iteration

            }

            // For Checkboxes and Radio Buttons,
            // We take the name and label of the group
            if (isset($field['type']) && ($field['type'] === "checkbox" || $field['type'] === "radio")) {

                // Check if the field was saved in options before
                if (isset($field['name']) && isset($options[$field['name']])) {
                    continue;
                }

                // Set option attributes
                $option = [
                    'name' => $field['name'],
                    'label' => $this->getFieldLabel($field, $withAlias, true), // Get groupLabel,
                ];

                // Save the option, by the name of the field
                $options[$option['name']] = $option;

                continue; // Skip the rest of the current loop iteration

            }

            $option = [
                'name' => $field['id'] ?? '',
                'label' => $this->getFieldLabel($field, $withAlias, false),
            ];

            // For buttons, replace label by value
            if (isset($field['type']) && in_array($field['type'], ['submit', 'reset', 'image', 'button'])) {
                $option['label'] = $field['value'] ?? $field['id'];
            }

            $options[$option['name']] = $option;
        }

        return array_values($options); // Remove keys
    }

    /**
     * Return Form Steps in rule actions format.
     * Required for rule builder
     *
     * @return array
     */
    public function getRuleSteps(): array
    {
        $builder = Json::decode($this->builder, true);
        $steps = $builder['settings']['formSteps']['fields']['steps']['value'];

        $options = array();

        if (count($steps) > 1) {
            foreach ($steps as $index => $title) {
                $option = [
                    'name' => $index,
                    'label' => $title,
                ];
                $options[$option['name']] = $option;
            }
        } else {
            $option = [
                'name' => 0,
                'label' => Yii::t('app', 'Same Step'),
            ];
            $options[$option['name']] = $option;
        }

        return $options;
    }

    /**
     * Return field ids as a simple array.
     * Required by rules engine
     *
     * @return array List of all input ids
     */
    public function getFieldIds(): array
    {
        $fields = Json::decode($this->fields, true);
        $fieldIDs = ArrayHelper::column($fields, 'id', 'id');
        return array_values($fieldIDs); // Only simple array
    }

    /**
     * Return All Fields, except buttons.
     * @return array
     */
    public function getFields(): array
    {
        $fields = Json::decode($this->fields, true);
        $fields = ArrayHelper::exclude($fields, ['submit', 'reset', 'image'], 'type');
        $fields = ArrayHelper::exclude($fields, null, 'data-repeater-id');
        $repeaterFields = $this->getRepeaterFields();
        return array_merge($fields, $repeaterFields);
    }

    /**
     * Return All Fields, except buttons.
     * Verifies that each field has a valid label
     *
     * Used by DataValidator
     * @return array
     */
    public function getFieldsForValidation(): array
    {
        $allFields = Json::decode($this->fields, true);
        $allFields = ArrayHelper::exclude($allFields, ['submit', 'reset', 'image'], 'type');
        $fields = [];
        foreach ($allFields as $field) {
            if (!isset($field['label'])) {
                $field['label'] = !empty($field['data-label']) ? $field['data-label'] : Yii::t('app', 'the input value');
            }
            if (isset($field['type']) && ($field['type'] == "checkbox" || $field['type'] == "radio") &&
                !isset($field['groupLabel'])) {
                $field['groupLabel'] = Yii::t('app', 'the input value');
            }
            $fields[] = $field;
        }

        return $fields;
    }

    /**
     * Return required fields.
     *
     * Used by this model
     *
     * @return array
     */
    public function getRequiredFields(): array
    {
        $allFields = Json::decode($this->fields, true);
        $allFields = ArrayHelper::exclude($allFields, ['submit', 'reset', 'image'], 'type');
        $requiredFields = ArrayHelper::filter($allFields, true, 'required');
        $fields = [];
        foreach ($requiredFields as $field) {
            if (!isset($field['label'])) {
                $field['label'] = !empty($field['data-label']) ? $field['data-label'] : Yii::t('app', 'the input value');
            }
            if (isset($field['type']) && ($field['type'] == "checkbox" || $field['type'] == "radio") &&
                !isset($field['groupLabel'])) {
                $field['groupLabel'] = Yii::t('app', 'the input value');
            }
            $fields[] = $field;
        }

        return $fields;
    }

    /**
     * Return unique fields.
     *
     * Used by DataValidator
     *
     * @return array
     */
    public function getUniqueFields(): array
    {
        $allFields = Json::decode($this->fields, true);
        $allFields = ArrayHelper::exclude($allFields, ['submit', 'reset', 'image'], 'type');
        $uniqueFields = ArrayHelper::filter($allFields, true, 'data-unique');
        $fields = [];
        foreach ($uniqueFields as $field) {
            if (!isset($field['label'])) {
                $field['label'] = !empty($field['data-label']) ? $field['data-label'] : Yii::t('app', 'the input value');
            }
            if (isset($field['type']) && ($field['type'] == "checkbox" || $field['type'] == "radio") &&
                !isset($field['groupLabel'])) {
                $field['groupLabel'] = Yii::t('app', 'the input value');
            }
            $fields[] = $field;
        }
        return $fields;
    }

    /**
     * Return All Fields, except files and buttons.
     * @return array
     */
    public function getFieldsWithoutFilesAndButtons(): array
    {
        $fields = Json::decode($this->fields, true);
        $fields = ArrayHelper::exclude($fields, ['submit', 'reset', 'image', 'file'], 'type');
        $fields = ArrayHelper::exclude($fields, null, 'data-repeater-id');
        $repeaterFields = $this->getRepeaterFields();
        return array_merge($fields, $repeaterFields);
//        $fields = Json::decode($this->fields, true);
//        return ArrayHelper::exclude($fields, ['submit', 'reset', 'image', 'file'], 'type');
    }

    /**
     * Return Fields without a disabled attribute, exclude buttons.
     * @return array
     */
    public function getEnabledFields(): array
    {
        return ArrayHelper::exclude($this->getFields(), true, 'disabled');
    }

    /**
     * Return File Fields.
     * Format: [name=>label]
     *
     * Used by Submissions App and AppController
     * @return array
     */
    public function getFileFields(): array
    {
        return ArrayHelper::filter($this->getFields(), 'file', 'type');
    }

    /**
     * Get Repeater Field Ids.
     * Format:
     * [
     *   [
     *     id => data-repeater-id
     *     defaultValues => [
     *       name => predefine-value,
     *     ]
     *   ]
     * ]
     *
     * Used by Form Widget
     * @return array
     */
    public function getRepeaterFields(): array
    {
        $fields = Json::decode($this->fields, true);
        $repeaterFields = ArrayHelper::group($fields, 'data-repeater-id');
        $fields = [];
        foreach ($repeaterFields as $id => $repeaterField) {
            $fields[] = [
                'id' => $id,
                'name' => $id,
                'label' => $repeaterField[0]['data-repeater-label'] ?? Yii::t('app', 'Repeater Field'),
                'alias' => $repeaterField[0]['data-repeater-alias'] ?? '',
                'min' => $repeaterField[0]['data-repeater-min'] ?? 1,
                'max' => $repeaterField[0]['data-repeater-max'] ?? 10,
                'fields' => $repeaterField,
                'defaultValues' => ArrayHelper::map($repeaterField, 'name', 'value'),
            ];
        }
        return $fields;
    }

    /**
     * Return fields for API
     * Exclude button fields
     *
     * @param bool $withAlias
     * @param bool $withFiles
     * @param bool $withDisabledFields
     * @return array
     */
    public function getFieldsForApi(bool $withAlias = false, bool $withFiles = false, bool $withDisabledFields = false): array
    {
        // All fields except buttons
        $fields = $this->getFields();
        // Exclude file fields
        if (!$withFiles) {
            $fields = ArrayHelper::exclude($fields, ['file'], 'type');
        }
        // Exclude disabled fields
        if (!$withDisabledFields) {
            $fields = ArrayHelper::exclude($fields, true, 'disabled');
        }

        $options = array();

        foreach ($fields as $field) {

            // For Matrix Fields
            if (!empty($field['data-matrix-id'])) {

                // Check if the field was saved in options before
                if (isset($field['name']) && isset($options[$field['name']])) {
                    continue;
                }

                // Set option attributes
                $option = [
                    'type' => $this->getFieldType($field),
                    'name' => !empty($field['name']) ? $field['name'] : null,
                    'alias' => !empty($field['alias']) ? $field['alias'] : null,
                    'label' => $this->getFieldLabel($field, $withAlias, false, true),
                    'answer' => null,
                ];

                // Save the option, by the name of the field
                $options[$option['name']] = $option;

                continue; // Skip the rest of the current loop iteration

            }

            // For Checkboxes and Radio Buttons,
            // We take the name and label of the group
            if (isset($field['type']) && ($field['type'] === "checkbox" || $field['type'] === "radio")) {

                // Check if the field was saved in options before
                if (isset($field['name']) && isset($options[$field['name']])) {
                    continue;
                }

                // Set option attributes
                $option = [
                    'type' => $this->getFieldType($field),
                    'name' => !empty($field['name']) ? $field['name'] : null,
                    'alias' => !empty($field['alias']) ? $field['alias'] : null,
                    'label' => $this->getFieldLabel($field, $withAlias, true), // Get groupLabel,
                    'answer' => null,
                ];

                // Save the option, by the name of the field
                $options[$option['name']] = $option;

                continue; // Skip the rest of the current loop iteration

            }

            // Default
            $option = [
                'type' => $this->getFieldType($field),
                'name' => !empty($field['name']) ? $field['name'] : null,
                'alias' => !empty($field['alias']) ? $field['alias'] : null,
                'label' => $this->getFieldLabel($field, $withAlias, false),
                'answer' => null,
            ];

            $options[$option['name']] = $option;
        }

        return array_values($options); // Remove keys
    }

    /**
     * Return fields for Submissions App
     * Exclude file and button fields
     *
     * @param bool $withAlias
     * @param bool $withFiles
     * @return array
     */
    public function getFieldsForSubmissions(bool $withAlias = false, bool $withFiles = false): array
    {
        // All fields except buttons and repeater fields
        $fields = $this->getFields();
        // Exclude file fields
        if (!$withFiles) {
            $fields = ArrayHelper::exclude($fields, ['file'], 'type');
        }
        // Exclude disabled fields
        $fields = ArrayHelper::exclude($fields, true, 'disabled');

        $options = array();

        foreach ($fields as $field) {

            // For Matrix Fields
            if (!empty($field['data-matrix-id'])) {

                // Check if the field was saved in options before
                if (isset($field['name']) && isset($options[$field['name']])) {
                    continue;
                }

                // Set option attributes
                $option = [
                    'name' => $field['name'],
                    'label' => $this->getFieldLabel($field, $withAlias, false, true),
                ];

                // Save the option, by the name of the field
                $options[$option['name']] = $option;

                continue; // Skip the rest of the current loop iteration

            }

            // For Checkboxes and Radio Buttons,
            // We take the name and label of the group
            if (isset($field['type']) && ($field['type'] === "checkbox" || $field['type'] === "radio")) {

                // Check if the field was saved in options before
                if (isset($field['name']) && isset($options[$field['name']])) {
                    continue;
                }

                // Set option attributes
                $option = [
                    'name' => $field['name'] ?? '',
                    'label' => $this->getFieldLabel($field, $withAlias, true), // Get groupLabel,
                ];

                // Save the option, by the name of the field
                $options[$option['name']] = $option;

                continue; // Skip the rest of the current loop iteration

            }

            // Default
            $option = [
                'name' => $field['id'] ?? '',
                'label' => $this->getFieldLabel($field, $withAlias, false),
            ];

            $options[$option['name']] = $option;
        }

        return array_values($options); // Remove keys
    }

    /**
     * Return an array of field names and labels
     *
     * Used by SubmissionEventHandler and AppController
     * To prepare the submission data for replacement token
     * That's why the alias feature is disabled by default
     *
     * @param bool $withAlias
     * @param bool $withFiles
     * @return array
     */
    public function getFieldsForEmail(bool $withAlias = false, bool $withFiles = true): array
    {
        $submissionFields = $this->getFieldsForSubmissions($withAlias, $withFiles);
        $fields = ArrayHelper::map($submissionFields, 'name', 'label');
        // Multi-Choice Fields Modifier: Labels
        foreach ($fields as $fieldName => $fieldLabel) {
            $parts = explode('_', $fieldName);
            if (!empty($parts)
                && in_array($parts[0], ['radio', 'checkbox', 'selectlist'])) {
                $fields[$fieldName . ':labels'] = Yii::t('app', 'Virtual Field') . ': ' . $fieldLabel . ' (' . Yii::t('app', 'Labels') . ')';
            }
        }
        return $fields;
    }

    /**
     * Get field list
     * Includes file fields, virtual fields, system fields, add-ons fields
     * Option to append aliases to the field label
     *
     * @param array $options
     * @return array
     */
    public function getFieldList(array $options = []): array
    {
        $defaultOptions = [
            'form' => true, // Include form fields
            'files' => true, // Include file fields
            'virtual' => false, // Include virtual fields
            'alias' => false, // Add alias to field labels
            'system' => false, // Include system fields
            'addons' => false, // Include fields added by add-ons
            'type' => 'associative', // associative: [name => label, name => label], multidimensional: [[name => $name, label => $label]]
        ];
        $options = array_merge($defaultOptions, $options);

        $fieldList = [];

        // Add Form Fields with Aliases and File fields
        if ($options['form']) {
            $fieldList = $this->getFieldsForSubmissions($options['alias'], $options['files']);
            if ($options['type'] === "associative") {
                $fieldList = ArrayHelper::map($fieldList, 'name', 'label');
            }
        }

        // Add Virtual Fields
        if ($options['virtual']) {
            if (count($fieldList) === count($fieldList, COUNT_RECURSIVE)) {
                foreach ($fieldList as $fieldName => $fieldLabel) {
                    $parts = explode('_', $fieldName);
                    if (!empty($parts)
                        && in_array($parts[0], ['radio', 'checkbox', 'selectlist'])) {
                        $fieldList[$fieldName . ':labels'] = Yii::t('app', 'Virtual Field') . ': ' . $fieldLabel . ' (' . Yii::t('app', 'Labels') . ')';
                    }
                }
            } else {
                foreach ($fieldList as $field) {
                    if (empty($field['name']) || empty($field['label'])) { continue; }
                    $parts = explode('_', $field['name']);
                    if (!empty($parts)
                        && in_array($parts[0], ['radio', 'checkbox', 'selectlist'])) {
                        $fieldList[] = [
                            'name' => $field['name'] . ':labels',
                            'label' => Yii::t('app', 'Virtual Field') . ': ' . $field['label'] . ' (' . Yii::t('app', 'Labels') . ')',
                        ];                    }
                }
            }
        }

        // Include System fields and Add-ons fields
        if ($options['system']) {
            if ($options['type'] === "associative") {
                $fieldList = SubmissionHelper::getFieldsForFieldMapping($fieldList, (bool) $options['addons']);
            } elseif ($options['type'] === "multidimensional") {
                $fields = SubmissionHelper::getFieldsForFieldMapping([], (bool) $options['addons']);
                foreach ($fields as $fieldName => $fieldLabel) {
                    $fieldList[] = [
                        'name' => $fieldName,
                        'label' => $fieldLabel,
                    ];
                }
            }
        }

        return $fieldList;
    }

    /**
     * Return All fields labels (name and label as assoc array).
     * Format: [name=>label]
     * In checkbox & radio elements: Replace by [name=>groupLabel]
     *
     * Used by FormController
     *
     * @param bool $withAlias Append alias to label when it's available
     * @return array
     */
    public function getLabels($withAlias = false)
    {
        // Name and Label of Provided Fields
        $fields = $this->getNameAndLabelOfProvidedFields($this->getFields(), $withAlias);
        return ArrayHelper::map($fields, 'name', 'label');
    }

    /**
     * Return labels of required fields (name and label as assoc array).
     * Format: [name=>label]
     * In checkbox & radio elements: Replace by [name=>groupLabel]
     *
     * Used by DataValidator
     *
     * @return array
     */
    public function getRequiredLabels(): array
    {
        // Required Fields
        $requiredFields = $this->getNameAndLabelOfProvidedFields($this->getRequiredFields());
        return ArrayHelper::column($requiredFields, 'label', 'name');
    }

    /**
     * Return All labels except of files and buttons. (name and label as assoc array).
     * Format: [name=>label]
     * In checkbox & radio elements: Replace by [name=>groupLabel]
     *
     * Used by FormController for export submissions as CSV file
     * @return array
     */
    public function getLabelsWithoutFilesAndButtons()
    {
        // Fields without files and buttons
        $fields = $this->getFieldsWithoutFilesAndButtons();
        // Get its Labels
        $allLabels = ArrayHelper::column($fields, 'label', 'name');
        // Get Checkboxes & Radio Buttons Labels
        $checkboxAndRadioLabels = ArrayHelper::column($fields, 'groupLabel', 'name');
        // Replace it with Checkboxes & Radio Buttons labels
        $labels = array_merge($allLabels, $checkboxAndRadioLabels);
        return $labels;
    }

    /**
     * Return field labels (name and label as an assoc array).
     * Format: [name=>label]
     * In checkbox & radio elements: Replace by [name=>groupLabel]
     *
     * Used by Form Report view
     *
     * @param bool $withAlias Append alias to the label when it's available
     * @return array
     */
    public function getLabelsForReport(bool $withAlias = false): array
    {
        $allFields = Json::decode($this->fields, true);
        $allFields = ArrayHelper::exclude($allFields, ['submit', 'reset', 'image', 'file'], 'type');
        $allFields = ArrayHelper::exclude($allFields, null, 'data-repeater-id');
        $fields = [];
        foreach ($allFields as $field) {
            if (!empty($field['id']) && substr($field['id'], 0, 6) !== 'matrix') {
                $fields[] = $field;
            }
        }
        // Name and Label of Provided Fields
        $fields = $this->getNameAndLabelOfProvidedFields($fields, $withAlias);
        return ArrayHelper::map($fields, 'name', 'label');

    }

    /**
     * Return Emails Fields.
     * Format: [name=>label]
     *
     * Used by FormSettings view
     *
     * @param bool $withAlias
     * @return array
     */
    public function getEmailLabels(bool $withAlias = true): array
    {
        $emailFields = ArrayHelper::filter($this->getFields(), 'email', 'type');
        $labels = array();
        foreach ($emailFields as $field) {
            $labels[$field['name']] = $this->getFieldLabel($field, $withAlias);
        }
        return $labels;
    }

    /**
     * Return File Field Labels.
     * Format: [name=>label]
     *
     * Used by Submissions App and AppController
     * @return array
     */
    public function getFileLabels(): array
    {
        $labels = [];
        $fields = $this->getFileFields();
        foreach ($fields as $field) {
            $labels[$field['name']] = isset($field['label']) && trim($field['label']) !== ''
                ? $field['label']
                : Yii::t('app', 'File Field with ID') . ": " . $field['id'];
        }
        return $labels;
    }

    /**
     * Return Required File Labels.
     * Format: [name=>label]
     *
     * Used by DataValidator
     *
     * @return array
     */
    public function getRequiredFileLabels(): array
    {
        $fileFields = ArrayHelper::filter($this->getFields(), 'file', 'type');
        $requiredFileFields = ArrayHelper::filter($fileFields, true, 'required');
        return ArrayHelper::column($requiredFileFields, 'label', 'name');
    }

    /**
     * Return Values of all options of all Select List of the form.
     * Format: [value1, value2, ...]
     *
     * Used by DataValidator
     *
     * @return array
     */
    public function getOptionValues(): array
    {
        $allFields = Json::decode($this->fields, true);
        $selects = ArrayHelper::filter($allFields, 'select', 'tagName');
        $options = [];
        foreach ($selects as $select) {
            if (isset($select['options'])) {
                foreach ($select['options'] as $option) {
                    $options[] = $option;
                }
            }
        }
        return array_values(ArrayHelper::column($options, 'value', 'value'));
    }

    /**
     * Return Values of all checkboxes of the form.
     * Format: [value1, value2, ...]
     *
     * Used by DataValidator
     *
     * @return array
     */
    public function getCheckboxValues(): array
    {
        $allFields = Json::decode($this->fields, true);
        $checkboxes = ArrayHelper::filter($allFields, 'checkbox', 'type');
        return array_values(ArrayHelper::column($checkboxes, 'value', 'id'));
    }

    /**
     * Return Values of all radio buttons of the form.
     * Format: [value1, value2, ...]
     *
     * Used by DataValidator
     *
     * @return array
     */
    public function getRadioValues(): array
    {
        $allFields = Json::decode($this->fields, true);
        $radios = ArrayHelper::filter($allFields, 'radio', 'type');
        return array_values(ArrayHelper::column($radios, 'value', 'id'));
    }

    /**
     * Get Field Choices
     *
     * Used by Limit Choices add-on
     *
     * @param string $fieldName Field Name
     * @return array
     */
    public function getFieldChoices(string $fieldName = ''): array
    {
        $parts = explode('_', $fieldName ?? '');
        $fieldType = $parts[0];
        $choices = [];
        if ($fieldType === 'selectlist') {
            $selects = ArrayHelper::filter($this->getFields(), 'select', 'tagName');
            $options = [];
            foreach ($selects as $select) {
                if (isset($select['name']) && $select['name'] === $fieldName && isset($select['options'])) {
                    foreach ($select['options'] as $option) {
                        $options[] = $option;
                    }
                }
            }
            $choices = ArrayHelper::column($options, 'label', 'value');
        } elseif ($fieldType === 'checkbox') {
            $checkboxes = ArrayHelper::filter($this->getFields(), 'checkbox', 'type');
            foreach ($checkboxes as $checkbox) {
                if (isset($checkbox['name']) && $checkbox['name'] === $fieldName) {
                    $value = $checkbox['value'] ?? null;
                    $label = $checkbox['label'] ?? null;
                    if (!is_null($value)) {
                        $choices[$value] = $label;
                    }
                }
            }
        } elseif ($fieldType === 'radio') {
            $radios = ArrayHelper::filter($this->getFields(), 'radio', 'type');
            foreach ($radios as $radio) {
                if (isset($radio['name']) && $radio['name'] === $fieldName) {
                    $value = $radio['value'] ?? null;
                    $label = $radio['label'] ?? null;
                    if (!is_null($value)) {
                        $choices[$value] = $label;
                    }
                }
            }
        }
        return $choices;
    }

    /**
     * Return ID of the first recaptcha component
     *
     * Used by a Form model
     *
     * @return null|string
     */
    public function getRecaptchaFieldID(): ?string
    {
        // Filter reCaptcha components
        $builder = Json::decode($this->builder, true);
        $recaptchaComponent = ArrayHelper::filter($builder['initForm'], 'recaptcha', 'name');
        // Get the first value of the array
        $component = array_shift($recaptchaComponent);
        return isset($component['fields']['id']) && ($component['fields']['id']['value'])
            ? $component['fields']['id']['value']
            : null;
    }

    /**
     * Return Radio Button Fields.
     * Format: [id=>value]
     *
     * Used by FormEventHandler
     * @return array
     */
    public function getRadioFields(): array
    {
        // Filter radio components
        $radios = ArrayHelper::filter($this->getFields(), 'radio', 'type');

        return ArrayHelper::column($radios, 'value', 'id');
    }

    /**
     * Return Select List Fields.
     * Format: [id=>value]
     *
     * Used by FormEventHandler
     * @return array
     */
    public function getSelectListFields(): array
    {
        $selects = ArrayHelper::filter($this->getFields(), 'select', 'tagName');
        $options = [];
        foreach ($selects as $select) {
            if (isset($select['options'])) {
                $i = 0;
                foreach ($select['options'] as $option) {
                    // Add field ID to each option
                    $option['id'] = $select['id'] . '_' . $i++;
                    $options[] = $option;
                }
            }
        }
        return ArrayHelper::column($options, 'value', 'id');
    }

    /**
     * Return Field IDs by container CSS class
     * Format: [id, id]
     *
     * Used by DataValidator
     * @param string $cssClass
     * @return array
     */
    public function getFieldsByContainerClass(string $cssClass = ''): array
    {
        $fields = [];
        if (is_string($cssClass) && !empty($cssClass)) {
            $builder = Json::decode($this->builder, true);
            foreach ($builder['initForm'] as $field) {
                if (isset($field['fields']['containerClass']['value'])) {
                    $containerClass = $field['fields']['containerClass']['value'];
                    if (count(array_filter(explode('.', $cssClass))) === count(array_intersect(array_filter(explode(' ', $containerClass)), array_filter(explode('.', $cssClass))))) {
                        if (isset($field['fields']['id']['value'])) {
                            $fields[] = $field['fields']['id']['value'];
                        }
                    }
                }
            }
        }
        return $fields;
    }

    /**
     * Get Field Information by input name
     *
     * Use by DataValidator
     * @param $fieldName
     * @return array
     */
    public function getFieldByName(string $fieldName): array
    {
        $fields = Json::decode($this->fields, true);
        $field = ArrayHelper::filter($fields, $fieldName, 'name');
        return $field;
    }


    /**
     * Return All fields aliases (name and alias as an assoc array).
     * Format: [name=>alias]
     *
     * Use by WebHooks Add-On
     * @return array
     */
    public function getAlias(): array
    {
        // Fields
        $fields = $this->getFields();
        // Alias
        return ArrayHelper::column($fields, 'alias', 'name');
    }

}
