<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2024 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\components\validators;

use app\components\rules\Finder;
use app\components\rules\RuleEngine;
use app\events\SubmissionErrorEvent;
use app\helpers\ArrayHelper;
use app\helpers\Html;
use app\helpers\Pager;
use app\models\FormData;
use app\models\FormRule;
use app\models\FormSubmission;
use Yii;
use yii\base\Component;
use yii\web\UploadedFile;

/**
 * Class DataValidator
 * @package app\components\validators
 */
class DataValidator extends Component
{

    /**
     * Properties
     */

    // Arrays
    /** @var array */
    public $data;                // FormSubmission Model Data property
    /** @var null|FormData @ */
    public $dataModel;           // FormData Model
    public $submissionModel;     // FormSubmission Model
    public $fields;              // All Fields
    public $radioValues;         // All Radio Buttons Values
    public $optionValues;        // All Option Values
    public $checkboxValues;      // All Checkboxes Values
    public $requiredFields;      // All Required Fields
    public $uniqueFields;        // All Unique Fields
    public $requiredFileLabels;  // All Required File Labels
    public $rules;               // FormRule Models
    public $files;               // $_FILES local copy

    /*
     * Private properties
     */
    private $errors;

    /**
     * @param FormSubmission $submissionModel
     * @throws \Exception
     */
    public function __construct(FormSubmission $submissionModel)
    {

        if (!isset($submissionModel->form_id) || !is_array($submissionModel->data)) {
            throw new \Exception(Yii::t('app', 'DataValidator needs the Form ID and Data attributes.'));
        }

        $this->submissionModel = $submissionModel;
        $this->data = $submissionModel->data;
        $this->files = $_FILES;
        $this->dataModel = $submissionModel->form->formData;
        $this->fields = $this->dataModel->getFieldsForValidation();
        $this->uniqueFields = $this->dataModel->getUniqueFields();
        $this->requiredFileLabels = $this->dataModel->getRequiredFileLabels();
        $this->requiredFields = $this->dataModel->getRequiredFields();
        $this->checkboxValues = $this->dataModel->getCheckboxValues();
        $this->radioValues = $this->dataModel->getRadioValues();
        $this->optionValues = $this->dataModel->getOptionValues();

        $formRuleModels = FormRule::findAll(['form_id' => $submissionModel->form_id, 'status' => FormRule::STATUS_ACTIVE]);
        /** @var $rules array Conditions and Actions of active rules */
        $this->rules = [];

        foreach ($formRuleModels as $formRuleModel) {
            $rule = [
                'conditions' => json_decode($formRuleModel['conditions']),
                'actions' => json_decode($formRuleModel['actions']),
                'opposite' => (boolean)$formRuleModel['opposite'],
            ];
            $this->rules[] = $rule;
        }

    }

    /**
     * Validates a data array.
     */
    public function validate()
    {

        // Unique Fields Validation
        $this->uniqueFieldsValidation();
        // Required Fields Validation
        $this->requiredFieldsValidation();
        // Validation by input types
        $this->fieldTypeValidation();

    }

    /**
     *
     */
    public function uniqueFieldsValidation()
    {

        if (is_array($this->uniqueFields) && !empty($this->uniqueFields)) {

            $this->trigger('beforeUniqueFieldsValidation');

            // Filter unique data
            $uniqueLabels = ArrayHelper::column($this->uniqueFields, 'label', 'name');
            $uniqueData = array_intersect_ukey($this->data, $uniqueLabels, function ($key1, $key2) {
                if (substr($key1, 0, 8) === 'repeater') {
                    if (isset($this->data[$key1]) && is_array($this->data[$key1])) {
                        $nestedFields = $this->data[$key1];
                        foreach ($nestedFields as $nestedField) {
                            $keys = array_keys($nestedField);
                            if (in_array($key2, $keys)) {
                                return 0;
                            }
                        }
                    }
                }
                if ($key2 == $key1) {
                    return 0;
                } else if ($key2 > $key1) {
                    return 1;
                }
                return -1;
            });

            foreach ($uniqueData as $name => $value) {
                if (!empty($value)) {
                    if (substr($name, 0, 8) === 'repeater') {
                        $duplicates = [];
                        foreach ($value as $nestedIndex => $nestedField) {
                            $uniqueNestedData = array_intersect_key($nestedField, $uniqueLabels);
                            foreach ($uniqueNestedData as $nestedName => $nestedValue) {
                                $nestedValue = is_array($nestedValue) ? implode(',', $nestedValue) : $nestedValue;
                                $nestedValue = trim($nestedValue);
                                if (!empty($nestedValue)) {
                                    $label = Yii::t('app', 'the input value');
                                    if (isset($uniqueLabels[$nestedName])) {
                                        $label = $uniqueLabels[$nestedName];
                                    }
                                    // Search fieldName:fieldValue in Request
                                    $searched = $nestedName . ':' . $nestedValue;
                                    if (in_array($searched, $duplicates)) {
                                        $event = new SubmissionErrorEvent();
                                        $event->message = Yii::t('app', '{attribute} "{value}" has already been taken.');
                                        $event->fieldName = $nestedName;
                                        $this->trigger(SubmissionErrorEvent::EVENT_NAME, $event);
                                        $this->addError($name . '[' . $nestedIndex . '][' . $nestedName . ']', $label, $nestedValue, $event->message);
                                    }
                                    $duplicates[] = $searched;
                                    // Search fieldName:fieldValue in DB
                                    $query = FormSubmission::find()
                                        ->where('form_id=:form_id', [':form_id' => $this->dataModel->form_id])
                                        ->andWhere(['like', 'data', '"' . $nestedName . '":"' . $nestedValue . '"']);
                                    if (!empty($this->submissionModel->id)) {
                                        $query->andWhere('id=:id', [':id' => $this->submissionModel->id]);
                                    }
                                    if ($query->count() > 0) {
                                        /** @var FormSubmission $submission */
                                        $submission = $query->one();
                                        // Only when we are not updating the form submission
                                        if ($this->submissionModel->id !== $submission->id) {
                                            $event = new SubmissionErrorEvent();
                                            $event->message = Yii::t('app', '{attribute} "{value}" has already been taken.');
                                            $event->fieldName = $nestedName;
                                            $this->trigger(SubmissionErrorEvent::EVENT_NAME, $event);
                                            $this->addError($name . '[' . $nestedIndex . '][' . $nestedName . ']', $label, $nestedValue, $event->message);
                                        }
                                    }
                                }
                            }
                        }
                    } else {
                        // Strip whitespace from the beginning and end of a string
                        $value = is_array($value) ? implode(',', $value) : $value;
                        $value = trim($value);
                        // Get label
                        $label = Yii::t('app', 'the input value');
                        if (isset($uniqueLabels[$name])) {
                            $label = $uniqueLabels[$name];
                        }
                        // Search fieldName:fieldValue in DB
                        $query = FormSubmission::find()
                            ->where('form_id=:form_id', [':form_id' => $this->dataModel->form_id])
                            ->andWhere(['like', 'data', '"' . $name . '":"' . $value . '"']);
                        if (!empty($this->submissionModel->id)) {
                            $query->andWhere('id=:id', [':id' => $this->submissionModel->id]);
                        }
                        if ($query->count() > 0) {
                            /** @var FormSubmission $submission */
                            $submission = $query->one();
                            // Only when we are not updating the form submission
                            if ($this->submissionModel->id !== $submission->id) {
                                $event = new SubmissionErrorEvent();
                                $event->message = Yii::t('app', '{attribute} "{value}" has already been taken.');
                                $event->fieldName = $name;
                                $this->trigger(SubmissionErrorEvent::EVENT_NAME, $event);
                                $this->addError($name, $label, $value, $event->message);
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * The required validation works in 3 steps:
     * 1. Get all required fields that are not in the submission (data and files).
     * If there is at least one, the validation fails.
     * 2. Get all required files.
     * If a required file is not in the global $_FILES, the validation fails.
     * 3. Get all required fields that pass the last steps and validate them with a RequiredValidator.
     */
    public function requiredFieldsValidation()
    {
        $requiredLabels = ArrayHelper::column($this->requiredFields, 'label', 'name');
        $conditionsAdapter = [];
        foreach ($this->data as $fieldName => $fieldValue) {
            $componentInfo = explode("_", $fieldName);
            $componentType = $componentInfo[0];
            if ($componentType === "checkbox") {
                // Replace Checkbox name by Checkbox ID
                $checkboxGroup = $this->dataModel->getFieldByName($fieldName);
                foreach ($checkboxGroup as $checkbox) {
                    // Field Value is the array of checked checkboxes
                    foreach ($fieldValue as $value) {
                        // Only add a checkbox if it's checked
                        if ($checkbox['value'] === $value) {
                            // Add Field ID instead of Field Name
                            $conditionsAdapter[$checkbox['id']] = $checkbox['value'];
                        }
                    }
                }
            } else {
                $conditionsAdapter[$fieldName] = $fieldValue;
            }
        }
        $actionsAdapter = [
            'toShow' => function ($data) {
            },
            'toHide' => function ($data) use (&$requiredLabels) {
                /**
                 * No validate a required field when it is hidden by a conditional rule
                 */
                /** @var Finder $data */
                if ($data->find("target") == "field") {
                    $field = $data->find(["target", "targetField"]);
                    $componentInfo = explode("_", $field);
                    $componentType = $componentInfo[0] ?? '';
                    // Update required matrix fields
                    if ($componentType === 'matrix') {
                        $requiredLabels = array_filter($requiredLabels, function ($key) use ($field) {
                            return strpos($key, $field) !== 0;
                        }, ARRAY_FILTER_USE_KEY);
                    }
                    // Update the rest of the required fields
                    if (isset($requiredLabels[$field])) unset($requiredLabels[$field]);
                    if (isset($this->requiredFileLabels[$field])) unset($this->requiredFileLabels[$field]);
                } elseif ($data->find("target") == "element") {
                    $element = $data->find(["target", "targetElement"]);
                    $fields = $this->dataModel->getFieldsByContainerClass($element);
                    foreach ($fields as $field) {
                        // Update required fields
                        if (isset($requiredLabels[$field])) unset($requiredLabels[$field]);
                        if (isset($this->requiredFileLabels[$field])) unset($this->requiredFileLabels[$field]);
                    }
                }
            },
            'toEnable' => function ($data) {
            },
            'toDisable' => function ($data) {
            },
            'performArithmeticOperations' => function ($data) {
            },
            'resetResult' => function ($data) {
            },
            'copy' => function ($data) {
            },
            'skip' => function ($data) use(&$requiredLabels) {
                /**
                 * Only validate a required field if it's present on the current page
                 */
                // Get the Paginated Form
                $pager = new Pager(Html::decode($this->dataModel->html));
                // Get Fields from previous pages except the current one
                $nextPage = isset($this->data['current_page']) ? $this->data['current_page'] + 1 : 0;
                /** @var Finder $data */
                $fields = Html::getFields(implode($pager->getPreviousPages((int) $data->find("step"), $nextPage)));
                // Update required fields
                foreach ($fields as $field) {
                    if (isset($requiredLabels[$field])) unset($requiredLabels[$field]);
                    if (isset($this->requiredFileLabels[$field])) unset($this->requiredFileLabels[$field]);
                }
            },
            'resetSkip' => function ($data) {
            },
            'formatNumber' => function ($data) {
            },
            'formatText' => function ($data) {
            }
        ];

        $oppositeAdapter = [
            'toShow' => "toHide",
            'toHide' => "toShow",
            'toEnable' => "toDisable",
            'toDisable' => "toEnable",
            'performArithmeticOperations' => "resetResult",
            'skip' => "resetSkip"
        ];

        /**
         * Rule Engine
         * Executes each conditional rule
         */
        foreach ($this->rules as $rule) {
            $engine = new RuleEngine($rule);
            $engine->run($conditionsAdapter, $actionsAdapter, $oppositeAdapter);
        }

        // Messages
        $requiredMessage = Yii::t('app', '{attribute} cannot be blank.');

        // Compares requiredLabels keys against data and files keys and returns the difference
        // All requiredLabel that are not in the submission data and file uploads
        // $requiredFields = array_diff_key($requiredLabels, $this->data, $this->files);
        $requiredFields = array_diff_ukey($requiredLabels, $this->data, $this->files, function ($key1, $key2) {
            if (substr($key2, 0, 8) === 'repeater') {
                if (isset($this->data[$key2]) && is_array($this->data[$key2])) {
                    $nestedFields = $this->data[$key2];
                    foreach ($nestedFields as $nestedField) {
                        $keys = array_keys($nestedField);
                        if (in_array($key1, $keys)) {
                            return 0;
                        }
                    }
                }
            }
            if ($key1 == $key2) {
                return 0;
            } else if ($key1 > $key2) {
                return 1;
            }
            return -1;
        });

        // If exist a requiredField, add a required error
        // Useful to validate if at least one checkbox of a group is checked
        if (count($requiredFields) > 0) {
            foreach ($requiredFields as $name => $label) {
                // If no file was uploaded
                $event = new SubmissionErrorEvent();
                $event->message = $requiredMessage;
                $event->fieldName = $name;
                $this->trigger(SubmissionErrorEvent::EVENT_NAME, $event);
                $this->addError($name, $label, '', $event->message);
            }
        }

        // Check all required File Inputs with $_FILES
        // Exclude of a step-by-step validation
        if (Yii::$app->controller->action->id !== "check") {
            foreach ($this->requiredFileLabels as $name => $label) {
                if (!is_array($this->files) || !isset($this->files[$name]) ||
                    !isset($this->files[$name]['name']) || empty($this->files[$name]['name'])) {
                    // If no file was uploaded
                    $event = new SubmissionErrorEvent();
                    $event->message = $requiredMessage;
                    $event->fieldName = $name;
                    $this->trigger(SubmissionErrorEvent::EVENT_NAME, $event);
                    $this->addError($name, $label, '', $event->message);
                }
            }
        }
        // Get all submission data that were not in the last validations
        $data = array_diff_key($this->data, $requiredFields, $this->requiredFileLabels);
        // Filter required data
        // $requiredData = array_intersect_key($data, $requiredLabels);
        $requiredData = array_intersect_ukey($data, $requiredLabels, function ($key1, $key2) {
            if (substr($key1, 0, 8) === 'repeater') {
                if (isset($this->data[$key1]) && is_array($this->data[$key1])) {
                    $nestedFields = $this->data[$key1];
                    foreach ($nestedFields as $nestedField) {
                        $keys = array_keys($nestedField);
                        if (in_array($key2, $keys)) {
                            return 0;
                        }
                    }
                }
            }
            if ($key2 == $key1) {
                return 0;
            } else if ($key2 > $key1) {
                return 1;
            }
            return -1;
        });
        // Check each field item with the requiredValidator
        $requiredValidator = new RequiredValidator();
        // If a field doesn't pass the validator, add a blank error
        if (count($requiredData) > 0) {
            foreach ($requiredData as $name => $value) {
                if (substr($name, 0, 8) === 'repeater') {
                    foreach ($value as $nestedIndex => $nestedField) {
                        $requiredNestedData = array_intersect_key($nestedField, $requiredLabels);
                        foreach ($requiredNestedData as $nestedName => $nestedValue) {
                            $nestedValue = is_array($nestedValue) ? implode(',', $nestedValue) : $nestedValue;
                            $requiredValidator->submissionModel = $this->submissionModel;
                            $requiredValidator->fieldName = $nestedName;
                            if (!$requiredValidator->validate($nestedValue, $error)) {
                                $this->addError($name . '[' . $nestedIndex . '][' . $nestedName . ']', $requiredLabels[$nestedName], '', $error);
                            }
                        }
                    }
                } else {
                    $value = is_array($value) ? implode(',', $value) : $value;
                    $requiredValidator->submissionModel = $this->submissionModel;
                    $requiredValidator->fieldName = $name;
                    if (!$requiredValidator->validate($value, $error)) {
                        $this->addError($name, $requiredLabels[$name], '', $error);
                    }
                }
            }
        }
    }

    /**
     * Validate form submission according to the field type and field settings
     *
     * @return void
     */
    public function fieldTypeValidation()
    {
        foreach ($this->fields as $field) {
            if (substr($field['name'], 0, 8) === 'repeater') {
                $data = $this->data[$field['data-repeater-id']] ?? [];
                foreach ($data as $nestedIndex => $nestedData) {
                    $field['data-name'] = $field['data-repeater-id'] . '[' . $nestedIndex . '][' . $field['name'] . ']';
                    $this->fieldSettingsValidation($field, $nestedData);
                }
            } else {
                $this->fieldSettingsValidation($field, $this->data, $this->files);
            }
        }
    }

    /**
     * Validate form fields according to the field settings
     *
     * @param array $field
     * @param array $data
     * @param array $files
     * @return void
     */
    public function fieldSettingsValidation(array $field, array $data, array $files = [])
    {
        // Messages
        $invalidMessage = "the input value has a not valid value.";

        foreach ($field as $key => $value) {
            // Text
            if ($key === "type" && $value === "text") {
                // Only when the input value is defined and is not empty
                if (isset($field["name"], $data[$field["name"]]) &&
                    (trim($data[$field["name"]]) !== "")) {
                    // A minlength or maxlength can be used
                    if (isset($field["minlength"]) || isset($field["maxlength"])) {
                        // Config string validator
                        $config = [
                            'submissionModel' => $this->submissionModel,
                            'fieldName' => $field["name"],
                        ];
                        // Add Min Length
                        if (isset($field["minlength"])) {
                            $config['min'] = $field["minlength"];
                        }
                        // Add Max Length
                        if (isset($field["maxlength"])) {
                            $config['max'] = $field["maxlength"];
                        }
                        $stringValidator = new StringValidator($config);
                        if (!$stringValidator->validate($data[$field["name"]], $error)) {
                            $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                        }
                    }
                    // A pattern can be used
                    if (isset($field["pattern"])) {
                        $regexValidator = new RegularExpressionValidator([
                            'pattern' => '/' . $field["pattern"] . '/', // Add regex delimiters
                            'submissionModel' => $this->submissionModel,
                            'fieldName' => $field["name"],
                        ]);
                        if (!$regexValidator->validate($data[$field["name"]], $error)) {
                            $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                        }
                    }
                }
            }
            // TextArea
            if ($key === "tagName" && $value === "textarea") {
                // Only when the input value is defined and is not empty
                if (isset($field["name"], $data[$field["name"]]) &&
                    (trim($data[$field["name"]]) !== "")) {
                    // A minlength or maxlength can be used
                    if (isset($field["minlength"]) || isset($field["maxlength"])) {
                        // Config string validator
                        $config = [
                            'submissionModel' => $this->submissionModel,
                            'fieldName' => $field["name"],
                        ];
                        // Add Min Length
                        if (isset($field["minlength"])) {
                            $config['min'] = $field["minlength"];
                        }
                        // Add Max Length
                        if (isset($field["maxlength"])) {
                            $config['max'] = $field["maxlength"];
                        }
                        $stringValidator = new StringValidator($config);
                        if (!$stringValidator->validate($data[$field["name"]], $error)) {
                            $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                        }
                    }
                }
            }
            // Tel
            if ($key === "type" && $value === "tel") {
                // Only when the input value is defined and is not empty
                if (isset($field["name"], $data[$field["name"]]) &&
                    (trim($data[$field["name"]]) !== "")) {
                    // A minlength or maxlength can be used
                    if (isset($field["minlength"]) || isset($field["maxlength"])) {
                        // Config string validator
                        $config = [
                            'submissionModel' => $this->submissionModel,
                            'fieldName' => $field["name"],
                        ];
                        // Add Min Length
                        if (isset($field["minlength"])) {
                            $config['min'] = $field["minlength"];
                        }
                        // Add Max Length
                        if (isset($field["maxlength"])) {
                            $config['max'] = $field["maxlength"];
                        }
                        $stringValidator = new StringValidator($config);
                        if (!$stringValidator->validate($data[$field["name"]], $error)) {
                            $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                        }
                    }
                    // A pattern can be used
                    if (isset($field["pattern"])) {
                        $regexValidator = new RegularExpressionValidator([
                            'pattern' => '/' . $field["pattern"] . '/', // Add regex delimiters
                            'submissionModel' => $this->submissionModel,
                            'fieldName' => $field["name"],
                        ]);
                        if (!$regexValidator->validate($data[$field["name"]], $error)) {
                            $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                        }
                    }
                }
            }
            // Url
            if ($key === "type" && $value === "url") {
                // Only when the input value is defined and is not empty
                if (isset($field["name"], $data[$field["name"]]) &&
                    (trim($data[$field["name"]]) !== "")) {
                    // A minlength or maxlength can be used
                    if (isset($field["minlength"]) || isset($field["maxlength"])) {
                        // config string validator
                        $config = [
                            'submissionModel' => $this->submissionModel,
                            'fieldName' => $field["name"],
                        ];
                        // Add Min Length
                        if (isset($field["minlength"])) {
                            $config['min'] = $field["minlength"];
                        }
                        // Add Max Length
                        if (isset($field["maxlength"])) {
                            $config['max'] = $field["maxlength"];
                        }
                        $stringValidator = new StringValidator($config);
                        if (!$stringValidator->validate($data[$field["name"]], $error)) {
                            $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                        }
                    }
                    // Config url validator
                    $config = [
                        'submissionModel' => $this->submissionModel,
                        'fieldName' => $field["name"],
                    ];
                    // A pattern can be used
                    if (isset($field["pattern"])) {
                        $config['pattern'] = '/' . $field["pattern"] . '/'; // Add regex delimiters
                    }
                    $urlValidator = new UrlValidator($config);
                    if (!$urlValidator->validate($data[$field["name"]], $error)) {
                        $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                    }
                }
            }
            // Color
            if ($key === "type" && $value === "color") {
                // Only when the input value is defined and is not empty
                if (isset($field["name"], $data[$field["name"]]) &&
                    (trim($data[$field["name"]]) !== "")) {
                    // hex color invalid
                    if (!preg_match('/^#[a-f0-9]{6}$/i', $data[$field["name"]])) {
                        $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $invalidMessage . ' ' .
                            Yii::t('app', 'It must be a hexadecimal color string (e.g. "#FFFFFF").'));
                    }
                }
            }
            // Password
            if ($key === "type" && $value === "password") {
                // Only when the input value is defined and is not empty
                if (isset($field["name"], $data[$field["name"]]) &&
                    (trim($data[$field["name"]]) !== "")) {
                    // A minlength or maxlength can be used
                    if (isset($field["minlength"]) || isset($field["maxlength"])) {

                        // Config string validator
                        $config = [
                            'submissionModel' => $this->submissionModel,
                            'fieldName' => $field["name"],
                        ];
                        // Add Min Length
                        if (isset($field["minlength"])) {
                            $config['min'] = $field["minlength"];
                        }
                        // Add Max Length
                        if (isset($field["maxlength"])) {
                            $config['max'] = $field["maxlength"];
                        }
                        $stringValidator = new StringValidator($config);
                        if (!$stringValidator->validate($data[$field["name"]], $error)) {
                            $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                        }
                    }
                    // A pattern can be used
                    if (isset($field["pattern"])) {
                        $regexValidator = new RegularExpressionValidator([
                            'pattern' => '/' . $field["pattern"] . '/', // Add regex delimiters
                            'submissionModel' => $this->submissionModel,
                            'fieldName' => $field["name"],
                        ]);
                        if (!$regexValidator->validate($data[$field["name"]], $error)) {
                            $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                        }
                    }
                }
            }
            // Email
            if ($key === "type" && $value === "email") {

                // Only when the input value is defined and is not empty
                if (isset($field["name"], $data[$field["name"]]) &&
                    (trim($data[$field["name"]]) !== "")) {
                    // A minlength or maxlength can be used
                    if (isset($field["minlength"]) || isset($field["maxlength"])) {

                        // Config string validator
                        $config = [
                            'submissionModel' => $this->submissionModel,
                            'fieldName' => $field["name"],
                        ];
                        // Add Min Length
                        if (isset($field["minlength"])) {
                            $config['min'] = $field["minlength"];
                        }
                        // Add Max Length
                        if (isset($field["maxlength"])) {
                            $config['max'] = $field["maxlength"];
                        }
                        $stringValidator = new StringValidator($config);
                        if (!$stringValidator->validate($data[$field["name"]], $error)) {
                            $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                        }
                    }

                    // Config email validator
                    $config = [
                        'submissionModel' => $this->submissionModel,
                        'fieldName' => $field["name"],
                    ];

                    // A pattern can be used
                    if (isset($field["pattern"])) {
                        $config['pattern'] = '/' . $field["pattern"] . '/'; // Add regex delimiters
                    }

                    // Whether to check if email's domain exists and has either an A or MX record.
                    // Be aware that this check can fail due to temporary DNS problems
                    // even if the email address is valid and an email would be deliverable
                    if (isset($field["data-check-dns"])) {
                        $config['checkDNS'] = true;
                    }

                    // Validate multiple emails separated by commas
                    if (isset($field["multiple"])) {
                        // Removes spaces
                        $emails = str_replace(" ", "", $data[$field["name"]]);
                        // Array of emails
                        $emails = explode(",", $emails);
                        if (count($emails) > 1) {
                            $config['message'] = Yii::t('app', '{attribute} has a invalid email format: Please use a comma to separate multiple email addresses.');
                        }
                        // Validate only one email address
                        $emailValidator = new EmailValidator($config);
                        foreach ($emails as $email) {
                            if (!$emailValidator->validate($email, $error)) {
                                $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                            }
                        }
                    } else {
                        // Validate only one email address
                        $emailValidator = new EmailValidator($config);

                        if (!$emailValidator->validate($data[$field["name"]], $error)) {
                            $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                        }

                    }
                }
            }
            // Radio
            if ($key === "type" && $value === "radio") {
                // Only when the input value is defined and not empty
                if (!empty($data[$field["name"]])) {
                    // If no values or if the received data does not match with the form data
                    if (empty($this->radioValues) || !in_array($data[$field["name"]], $this->radioValues)) {
                        $this->addError($field["data-name"] ?? $field["name"], $field["groupLabel"], '', $invalidMessage);
                    }
                }
            }
            // Checkbox
            if ($key === "type" && $value === "checkbox") {
                // Only when the input value is not empty
                if (!empty($data[$field["name"]])) {
                    // If no values or if the received data does not match with the form data
                    foreach ($data[$field["name"]] as $labelChecked) {
                        if (empty($this->checkboxValues) || !in_array($labelChecked, $this->checkboxValues)) {
                            $this->addError($field["data-name"] ?? $field["name"], $field["groupLabel"], '', $invalidMessage);
                        }
                    }
                }
            }
            // Select List
            if ($key === "tagName" && $value === "select") {
                // Only when the input value is not empty
                // if (isset($field["name"], $data[$field["name"]]) && is_array($data[$field["name"]]) && !empty($data[$field["name"]])) {
                //     // If no labels or if the received data does not match with the form data
                //     foreach ($data[$field["name"]] as $optionSelected) {
                //         if (empty($this->optionValues) || !in_array($optionSelected, $this->optionValues)) {
                //             $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $invalidMessage);
                //         }
                //     }
                // }
            }
            // Number & Range
            if (($key === "type" && $value === "number") || ($key === "type" && $value === "range")) {
                // Only when the input value is defined and is not empty
                if (isset($field["name"], $data[$field["name"]]) &&
                    (trim($data[$field["name"]]) !== "")) {

                    // Config number validator
                    $config = [
                        'submissionModel' => $this->submissionModel,
                        'fieldName' => $field["name"],
                    ];
                    // Min Number Validation (Minimum value required)
                    if (isset($field["min"])) {
                        $config['min'] = $field["min"];
                    }

                    // Max Number Validation (Maximum value required)
                    if (isset($field["max"])) {
                        $config['max'] = $field["max"];
                    }

                    // Only Integer Validation (Whether the attribute value can only be an integer)
                    if (isset($field["data-integer-only"])) {
                        $config['integerOnly'] = true;
                    }

                    // Pattern to Validate only Integer Numbers (The regular expression for matching integers)
                    if (isset($field["data-integer-pattern"])) {
                        $config['integerPattern'] = $field["data-integer-pattern"];
                    }

                    // Default Pattern to validate the numbers. Can be overwritten by a data number pattern
                    if (isset($field["pattern"])) {
                        $config['numberPattern'] = $field["pattern"];
                    }

                    // Pattern to Validate the Number (The regular expression for matching numbers)
                    if (isset($field["data-number-pattern"])) {
                        $config['numberPattern'] = $field["data-number-pattern"];
                    }

                    $numberValidator = new NumberValidator($config);

                    if (!$numberValidator->validate($data[$field["name"]], $error)) {
                        $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                    }
                }
            }
            // Date & DateTime & Time & Month & Week
            if (($key === "type" && $value === "date") || ($key === "type" && $value === "datetime-local") ||
                ($key === "type" && $value === "time") || ($key === "type" && $value === "month") ||
                ($key === "type" && $value === "week")) {
                // Only when the input value is defined and is not empty
                if (isset($field["name"], $data[$field["name"]]) &&
                    (trim($data[$field["name"]]) !== "")) {

                    // DateValidator Configuration array
                    $config = [
                        'submissionModel' => $this->submissionModel,
                        'fieldName' => $field["name"],
                    ];

                    // Date Format by default
                    $format = "Y-m-d";
                    // Change Format
                    if ($value === "datetime-local") {
                        // DateTime Format
                        $format = "Y-m-d\TH:i"; // "Y-m-d H:i";
                    } elseif ($value === "time") {
                        // Time Format
                        $format = "H:i:s";
                        $timeValues = explode(':', $data[$field["name"]]);
                        $seconds = count($timeValues) < 3 ? ':00' : '';
                        $data[$field["name"]] = $data[$field["name"]] . $seconds;
                    } elseif ($value === "month") {
                        // Month Format
                        $format = "Y-m";
                    } elseif ($value === "week") {
                        // First, validate by regular expression
                        $regexValidator = new RegularExpressionValidator([
                            'pattern' => "/\d{4}-W\d{2}/",
                            'submissionModel' => $this->submissionModel,
                            'fieldName' => $field["name"],
                        ]);
                        if (!$regexValidator->validate($data[$field["name"]], $error)) {
                            $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                        }
                        // Next, convert to date, to dateValidator (min / max)
                        if (isset($field["min"])) {
                            $config['tooSmall'] = Yii::t('app', '{attribute} must be no less than {weekMin}.', [
                                'weekMin' => $field["min"],
                            ]);
                            $field["min"] = date("Y-m-d", strtotime($field["min"]));
                        }
                        if (isset($field["max"])) {
                            $config['tooBig'] = Yii::t('app', '{attribute} must be no greater than {weekMax}.', [
                                'weekMax' => $field["max"],
                            ]);
                            $field["max"] = date("Y-m-d", strtotime($field["max"]));
                        }
                        $data[$field["name"]] = date("Y-m-d", strtotime($data[$field["name"]]));
                    }

                    // Add a PHP format
                    $config['format'] = "php:" . $format;

                    // Add Min Date Validation (The value must be later than this option)
                    if (isset($field["min"])) {
                        preg_match("/{(.*?)(?::(.+?))?}/", $field["min"], $matches);
                        if ($matches && !empty($matches[1])) {
                            $tag = strtolower($matches[1]);
                            $modifier = !empty($matches[2]) ? (int)$matches[2] : null;
                            if ($tag === 'today') {
                                $field["min"] = date($format);
                                if ($modifier) {
                                    $field["min"] = date('Y-m-d', strtotime($field["min"] . ' ' . $modifier . ' day'));
                                }
                            }
                        }
                        $config['min'] = $field["min"];
                    }

                    // Add Max Date Validation (The value must be earlier than this option)
                    if (isset($field["max"])) {
                        preg_match("/{(.*?)(?::(.+?))?}/", $field["max"], $matches);
                        if ($matches && !empty($matches[1])) {
                            $tag = strtolower($matches[1]);
                            $modifier = !empty($matches[2]) ? (int)$matches[2] : null;
                            if ($tag === 'today') {
                                $field["max"] = date($format);
                                if ($modifier) {
                                    $field["max"] = date('Y-m-d', strtotime($field["max"] . ' ' . $modifier . ' day'));
                                }
                            }
                        }
                        $config['max'] = $field["max"];
                    }

                    $dateValidator = new DateValidator($config);

                    if (!$dateValidator->validate($data[$field["name"]], $error)) {
                        $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                    }
                }
            }
            // File
            if ($key === "type" && $value === "file") {
                // Only when the $_FILES name value is not empty
                if (!empty($files[$field["name"]]['name'])) {

                    // Config FileValidator
                    $config = [
                        'submissionModel' => $this->submissionModel,
                        'fieldName' => $field["name"],
                    ];

                    // File type validation
                    // Note that you should enable FileInfo PHP extension.
                    if (isset($field["accept"]) && extension_loaded('fileinfo')) {
                        // Removes dots
                        $extensions = str_replace(".", "", $field["accept"]);
                        // Removes spaces
                        $extensions = str_replace(" ", "", $extensions);
                        $config['extensions'] = explode(",", $extensions);
                        // All file name extensions are allowed
                        $config['extensions'] = empty($extensions) ? null : $config['extensions'];
                    }

                    // File Min Size validation
                    if (isset($field["data-min-size"])) {
                        // Removes dots
                        $config['minSize'] = (int)$field["data-min-size"];
                    }

                    // File Max Size validation
                    if (isset($field["data-max-size"])) {
                        // Removes dots
                        $config['maxSize'] = (int)$field["data-max-size"];
                    }

                    // Min Files validation
                    if (isset($field["data-min-files"])) {
                        // Removes dots
                        $config['minFiles'] = (int)$field["data-min-files"];
                    }

                    // Max Files validation
                    if (isset($field["data-max-files"])) {
                        // Removes dots
                        $config['maxFiles'] = (int)$field["data-max-files"];
                    }

                    $file = UploadedFile::getInstancesByName($field["name"]);

                    $fileValidator = new FileValidator($config);

                    if (!$fileValidator->validate($file, $error)) {
                        $this->addError($field["data-name"] ?? $field["name"], $field["label"], '', $error);
                    }
                }
            }
        }
    }

    /**
     * Returns a value indicating is there are any validation error.
     * @param string|null $attribute attribute name. Use null to check all attributes.
     * @return boolean is there are any error.
     */
    public function hasErrors(?string $attribute = null): bool
    {
        return $attribute === null ? !empty($this->errors) : isset($this->errors[$attribute]);
    }

    /**
     * Returns errors for all attributes or single attribute.
     * @param string|null $attribute attribute name. Use null to retrieve errors for all attributes.
     * @return array errors for all attributes or the specified attribute. Empty array is returned if no error.
     * Note that when returning errors for all attributes, the result is a two-dimensional array, like the following:
     *
     * ~~~
     * [
     *     'inputtext-0' => [
     *         'Username is required.',
     *         'Username must contain only word characters.',
     *     ],
     *     'inputtext-1' => [
     *         'Email address is invalid.',
     *     ]
     * ]
     * ~~~
     *
     * @property array An array of errors for all attributes. Empty array is returned if no error.
     * The result is a two-dimensional array. See [[getErrors()]] for detailed description.
     */
    public function getErrors(?string $attribute = null): array
    {
        if ($attribute === null) {
            return $this->errors === null ? [] : $this->getUniqueErrors();
        } else {
            return $this->errors[$attribute] ?? [];
        }
    }

    /**
     * Adds a new error to the specified attribute.
     *
     * @param $attribute
     * @param string $label
     * @param string $value
     * @param string $error
     */
    public function addError($attribute, string $label = '', string $value = '', string $error = '')
    {
        if (!empty($label) && !empty($value)) {
            $this->errors[$attribute][] = Yii::t('app', $error, [
                'attribute' => $label,
                'value' => $value,
            ]);
        } else {
            $errorMessage = str_replace(Yii::t('app', 'the input value'), $label, $error);
            $errorMessage = str_replace('{attribute}', $label, $errorMessage);
            $this->errors[$attribute][] = $errorMessage;
        }
    }

    /**
     * Get Unique Errors
     *
     * @return array
     */
    public function getUniqueErrors(): array
    {
        $errors = [];
        foreach ($this->errors as $key => $value) {
            $errors[$key] = array_unique($value);
        }
        return $errors;
    }
}
